<?php
/**
 * Mercadopago Payments Module for Prestashop
 *
 * Tested in Prestashop v1.4.3.0, 1.4.11.0, 1.5.0.17, 1.5.6.2, 1.6.0.5, 1.6.0.14, 1.6.1.0
 *
 * @author    Rinku Kazeno <development@kazeno.co>
 *
 * @copyright Copyright (c) 2012-2015, Rinku Kazeno
 * @license   This module is licensed to the user, upon purchase
 *   from either Prestashop Addons or directly from the author,
 *   for use on a single commercial Prestashop install, plus an
 *   optional separate non-commercial install (for development/testing
 *   purposes only). This license is non-assignable and non-transferable.
 *   To use in additional Prestashop installations an additional
 *   license of the module must be purchased for each one.

 *   The user may modify the source of this module to suit their
 *   own business needs, as long as no distribution of either the
 *   original module or the user-modified version is made.
 *
 * @version 2.19
 */

if (!defined('_PS_VERSION_'))
    exit;

class Mercadopago extends PaymentModule
{
    const MODULE_NAME = 'mercadopago';                  //DON'T CHANGE!!
    const CONFIG_PREFIX = 'MERCADOPAGO';    //prefix for all internal config constants
    const DB_TABLE = 'mercadopago_orders';
    const LOCK_FILE = 'lock/validator.lock';
    const SERVER_IP = '216.33';
    protected $countryCurrency = array( 'mla'=>'ars', 'mlb'=>'brl', 'mco'=>'cop', 'mlc'=>'clp', 'mlm'=>'mxn','mlv'=>'vef');
    protected $countryName = array( 'mla'=>'Argentina', 'mlb'=>'Brasil', 'mlc'=>'Chile', 'mco'=>'Colombia', 'mlm'=>'Mexico', 'mlv'=>'Venezuela');
    protected $paymentsArray = array(
        //'account_money',
        'credit_card',
        'debit_card',
        'ticket',
        'bank_transfer',
        'atm'
    );
    public $delta = 1;          //Maximum amount of permissible discrepancy between Order amount and amount payed by Customer, in Currency Units
    public $transactionId;
    public $warnings = array();
    public $fatalWarnings = array();
    public $wait_status = '';

    public function __construct()
	{
        $this->name = 'mercadopago';                  //DON'T CHANGE!!
		$this->tab = 'payments_gateways';
		$this->version = '2.19';
		$this->author = 'R. Kazeno';
        $this->need_instance = 1;
        $this->ps_versions_compliancy = array('min' => '1.4', 'max' => '1.6');
        $this->module_key = 'b4f53a8ae2244d9c8f4579f2447b70e3';
        $this->currencies = TRUE;
        $this->currencies_mode = 'radio';
        $this->bootstrap = true;
        
        parent::__construct();
        $this->displayName = 'Mercadopago';
		$this->description = $this->l('Accept Mercadopago payments using the IPN 2.0 API');
        $this->confirmUninstall = $this->l('This will delete any configured settings for this module. Continue?');
        $shopCurrency = $this->getCurrency(Configuration::get('PS_CURRENCY_DEFAULT'));
        if (!is_object($shopCurrency)) {            //Assume default currency if none registered
            $shopCurrency = NEW Currency(Configuration::get('PS_CURRENCY_DEFAULT'));
        }
        if (!function_exists('fsockopen'))          //Check if fsockopen connections are not blocked by host
            $this->fatalWarnings[] = $this->l('Your host has disabled fsockopen. Contact their technical support to have them enable outbound connections for your account.');
        if (!extension_loaded('openssl'))
            $this->fatalWarnings[] = $this->l('You have the Openssl PHP extension disabled. This module requires external SSL connections.');
        if (!extension_loaded('json'))
            $this->fatalWarnings[] = $this->l('JSON extension not installed. This module requires it for handling JSON-encoded data.');
        if (Configuration::get(self::CONFIG_PREFIX.'_CLIENT_ID') != 0 AND !in_array(Tools::strtolower($shopCurrency->iso_code), $this->countryCurrency))
            $this->fatalWarnings[] = $this->l('Selected currency not supported by Mercadopago. Please change it in your Currency Restrictions configuration.');
        if (Configuration::get(self::CONFIG_PREFIX.'_CLIENT_ID')=='0' OR !Tools::strlen(Configuration::get(self::CONFIG_PREFIX.'_CLIENT_ID')))
            $this->fatalWarnings[] = $this->l('You need to configure your account settings.');
        if (Configuration::get(self::CONFIG_PREFIX.'_SANDBOX_MODE'))
            $this->warnings[] = $this->l('Sandbox mode is currently enabled. The store will not process real customer payments until it\'s disabled.');
        if (!Configuration::get('PS_SHOP_ENABLE'))
            $this->warnings[] = $this->l('Your shop is currently in maintenance mode. Mercadopago automatic payment notifications will not work until it is disabled.');
        if (0777 !== (fileperms(_PS_MODULE_DIR_."{$this->name}/".self::LOCK_FILE) & 0777))
            $this->warnings[] = $this->l('The file').' "'._PS_MODULE_DIR_."{$this->name}/".self::LOCK_FILE.'" '.$this->l('needs full permissions (777) for the module to operate correctly. Please modify them from your ftp client.');
        $this->warning = implode(' | ', ($this->warnings+$this->fatalWarnings));
        if (file_exists(_PS_MODULE_DIR_."{$this->name}/customizations.php"))            //Load customizations class
            require_once _PS_MODULE_DIR_."{$this->name}/customizations.php";
        else
            require_once _PS_MODULE_DIR_."{$this->name}/override/customizations.php";
        if (_PS_VERSION_ < '1.5') {
            if (Tools::strlen(Configuration::get(self::CONFIG_PREFIX . '_MERCHANT_ID'))) {       //Force upgrade to API 2.0 on older PS versions
                require_once _PS_ROOT_DIR_ . "/modules/{$this->name}/upgrade/Upgrade-1.28.php";
                upgrade_module_1_28($this);
            }
            if (!Tools::strlen(Configuration::get(self::CONFIG_PREFIX . '_FEE'))) {       //Force upgrade to version 2.05 on older PS versions
                require_once _PS_ROOT_DIR_ . "/modules/{$this->name}/upgrade/Upgrade-2.05.php";
                upgrade_module_2_05($this);
            }
            //Backwards compatibility for Prestashop versions < 1.5
            require(_PS_MODULE_DIR_ . $this->name . '/backward_compatibility/backward.php');
            if (!in_array('HelperForm', get_declared_classes()))
                require(_PS_MODULE_DIR_ . $this->name . '/backward_compatibility/HelperForm.php');
        }
    }
    
    public function install()
    {
        $mails = array(
            'en/waiting_for_payment.txt',
            'en/waiting_for_payment.html',
            'es/waiting_for_payment.txt',
            'es/waiting_for_payment.html'
        );
        if (!defined('_PS_HOST_MODE_')) {
            foreach ($mails as $file) {             //install pending status mail templates
                if (!file_exists(_PS_MAIL_DIR_ . $file) && is_dir(_PS_MAIL_DIR_ . Tools::substr($file, 0, 2))) {
                    @copy(_PS_MODULE_DIR_ . "{$this->name}/mails/{$file}", _PS_MAIL_DIR_ . $file);
                }
            }
        }
        //Bypass georestrictions for Mercadopago notification server block
        if (!in_array(self::SERVER_IP, explode(';', Configuration::get('PS_GEOLOCATION_WHITELIST'))))
            Configuration::updateValue('PS_GEOLOCATION_WHITELIST', Configuration::get('PS_GEOLOCATION_WHITELIST').';'.self::SERVER_IP);

        @chmod(_PS_MODULE_DIR_."{$this->name}/".self::LOCK_FILE, 0777);
        @copy(_PS_MODULE_DIR_."{$this->name}/override/customizations.php", _PS_MODULE_DIR_."{$this->name}/customizations.php");
        $db = Db::getInstance();
        return (
            parent::install() AND
            $db->Execute(
                'CREATE TABLE IF NOT EXISTS `' . pSQL(_DB_PREFIX_.self::DB_TABLE) . '` (
                `transaction_id` varchar(20) NOT NULL,
                `order_id` int(10) NULL,
                `cart_id` int(10) NOT NULL UNIQUE,
                `fee` varchar(10) NULL,
                PRIMARY KEY (`transaction_id`)
                )') AND
            $this->registerHook('payment') AND
            $this->registerHook('paymentReturn') AND
            $this->registerHook('invoice') AND
            Configuration::updateValue(self::CONFIG_PREFIX.'_CLIENT_ID', '0') AND
            Configuration::updateValue(self::CONFIG_PREFIX.'_CLIENT_SECRET', '') AND
            Configuration::updateValue(self::CONFIG_PREFIX.'_TOKEN_KEY', '') AND
            Configuration::updateValue(self::CONFIG_PREFIX.'_TOKEN_DATE', '') AND
            Configuration::updateValue(self::CONFIG_PREFIX.'_COUNTRY', 'mla') AND
            Configuration::updateValue(self::CONFIG_PREFIX.'_PAYMENT_METHODS', '["account_money","credit_card","debit_card","ticket","bank_transfer","atm"]') AND
            Configuration::updateValue(self::CONFIG_PREFIX.'_INSTALLMENTS', '') AND
            Configuration::updateValue(self::CONFIG_PREFIX.'_CHECKOUT_TYPE', 'modal') AND
            Configuration::updateValue(self::CONFIG_PREFIX.'_SANDBOX_MODE', '0') AND
            Configuration::updateValue(self::CONFIG_PREFIX.'_FEE', '0.00%') AND
            Configuration::updateValue(self::CONFIG_PREFIX.'_WAIT_STATUS', '0')
        );
    }
    
    public function uninstall()
    {
        return (parent::uninstall() AND 
            Configuration::deleteByName(self::CONFIG_PREFIX.'_CLIENT_ID') AND
            Configuration::deleteByName(self::CONFIG_PREFIX.'_CLIENT_SECRET') AND
            Configuration::deleteByName(self::CONFIG_PREFIX.'_TOKEN_KEY') AND
            Configuration::deleteByName(self::CONFIG_PREFIX.'_TOKEN_DATE') AND
            Configuration::deleteByName(self::CONFIG_PREFIX.'_COUNTRY') AND
            Configuration::deleteByName(self::CONFIG_PREFIX.'_PAYMENT_METHODS') AND
            Configuration::deleteByName(self::CONFIG_PREFIX.'_INSTALLMENTS') AND
            Configuration::deleteByName(self::CONFIG_PREFIX.'_CHECKOUT_TYPE') AND
            Configuration::deleteByName(self::CONFIG_PREFIX.'_SANDBOX_MODE') AND
            Configuration::deleteByName(self::CONFIG_PREFIX.'_FEE') AND
            Configuration::deleteByName(self::CONFIG_PREFIX.'_WAIT_STATUS')
        );
    }
    
    
    public function getContent()
    {
        require_once 'API/mercadopagoAPI.php';

        ob_start();
            if (Tools::isSubmit('submitMercadopago'))
                echo $this->_getErrors() ? $this->_getErrors() : $this->_saveConfig();
            if (Configuration::get(self::CONFIG_PREFIX.'_CLIENT_ID') != '0' AND Tools::strlen(Configuration::get(self::CONFIG_PREFIX.'_CLIENT_ID'))) {
                //Check IPN Access is working correctly
                try {
                    $this->_getUpdatedToken(TRUE) OR $this->displayError($this->l('Invalid response from Mercadopago authentication server'));
                } catch (Exception $e) {
                    echo $this->displayError($e->getMessage());
                }
            }
        $header = ob_get_clean();
        $currentCountry = Tools::getValue('country', Configuration::get(self::CONFIG_PREFIX.'_COUNTRY'));
        $currentCurrency = $this->getCurrency();
        $payments = $this->toolGetPayments();
        $statuses = OrderState::getOrderStates($this->context->language->id);
        $selectableStatuses = array();
        foreach ($statuses as $stat) {
            if ($stat['id_order_state'] > 9) {
                $selectableStatuses[] = array('value' => $stat['id_order_state'], 'text' => $stat['name']);
            }
        }
        $waitStatus = $this->wait_status ? $this->wait_status : Tools::getValue('wait_status', Configuration::get(self::CONFIG_PREFIX.'_WAIT_STATUS'));
        switch ($this->context->language->iso_code) {
            case 'es':
            case 'ar':
            case 'mx':
            case 'cl':
            case 'co':
            case 've':
                $doclang = 'es_AR';
                break;
            case 'pt':
            case 'br':
                $doclang = 'pt_BR';
                break;
            default:
                $doclang = 'en_US';
                break;
        }
        $dataUrl = 'https://www.mercadopago.com/' . $currentCountry . ($currentCountry == 'mlb' ? '/ferramentas/aplicacoes' : '/herramientas/aplicaciones');
        $dataUrl2 = 'https://www.mercadopago.com/' . $currentCountry . ($currentCountry == 'mlb' ? '/ferramentas/notificacoes' : '/herramientas/notificaciones');
        ob_start();
            echo $this->l('You need to configure the IPN address in your Mercadopago account. Copy the following address into your clipboard:'); ?><br /><br />
            <b style="display: block; text-align: center;"><?php echo (method_exists('Tools','getShopDomainSsl') ? Tools::getShopDomainSsl(true, true) : Tools::getHttpHost(true, true)) .      //1.3.x Hack
                    __PS_BASE_URI__ . (_PS_VERSION_ < '1.5' ? "modules/{$this->name}/validation.php" : "index.php?fc=module&module={$this->name}&controller=validation"); ?></b><br />
            <?php echo $this->l('Now open the following link, which will take you to your Mercadopago IPN settings page, and paste the above address you just copied into the "Notification URL" field and click "Save":'); ?><br />
            <a id="mp_config_link2" href="<?php  echo $dataUrl2;  ?>" target="_blank" style="text-decoration: underline; font-weight: 700;"><?php  echo $dataUrl2;  ?></a>
            <?php echo $this->l('(opens in a new window)');
        $notifBox = ob_get_clean();
        ob_start(); ?>
        <script>
            (function() {
                if (typeof jQuery.fn.prop != 'function') {
                    jQuery.fn.prop = jQuery.fn.attr;
                }
                var link = $("#mp_config_link");
                var link2 = $("#mp_config_link2");
                var sel = $("#country");
                var sel2 = $("#wait_status");
              <?php if (_PS_VERSION_ < '1.6') : ?>
                var waitname = $('#wait_status_name').parent().prev().add($('#wait_status_name').parent());
                var waitcolor = $('.mColorPickerInput').parent().prev().add($('.mColorPickerInput').parent());
              <?php else : ?>
                var waitname = $('#wait_status_name').parent().parent();
                var waitcolor = waitname.next('.form-group');
              <?php endif; ?>
                waitname.hide().prop('disabled', true);
                waitcolor.hide().prop('disabled', true);
                function changeUrls() {
                    var country = sel.val();
                    var url = country == 'mlb' ? '/ferramentas/aplicacoes' : '/herramientas/aplicaciones';
                    link.html('https://www.mercadopago.com/'+country+url).attr('href', 'https://www.mercadopago.com/'+country+url);
                    var url2 = country == 'mlb' ? '/ferramentas/notificacoes' : '/herramientas/notificaciones';
                    link2.html('https://www.mercadopago.com/'+country+url2).attr('href', 'https://www.mercadopago.com/'+country+url2);
                }
                function toggleNewStat() {
                    if (sel2.val()=='new') {
                        waitname.show().prop('disabled', false);
                        waitcolor.show().prop('disabled', false);
                    } else {
                        waitname.hide().prop('disabled', true);
                        waitcolor.hide().prop('disabled', true);
                    }
                }
                sel.bind("change", changeUrls);
                sel2.bind("change", toggleNewStat);
            })();
        </script>
        <?php $guijs = ob_get_clean();

        $fields_form = array(
            array(
                'form' => array(
                    'legend' => array(
                        'title' => '1. '.$this->l('Account')
                    ),
                    'input' => array(
                        array(
                            'type' => 'select',
                            'label' => $this->l('Country'),
                            'name' => 'country',
                            'options' => array(
                                'query' => array(
                                    array('value' =>'mla', 'text' =>'Argentina'),
                                    array('value' =>'mlb', 'text' =>'Brasil'),
                                    array('value' =>'mlc', 'text' =>'Chile'),
                                    array('value' =>'mco', 'text' =>'Colombia'),
                                    array('value' =>'mlm', 'text' =>'México'),
                                    array('value' =>'mlv', 'text' =>'Venezuela'),
                                ),
                                'id' => 'value',
                                'name' => 'text'
                            ),
                            'desc' => $this->l('After selecting your country above and logging into your Mercadopago account in a different browser window, you\'ll find the info you need at the following address (opens in a new window):')." <a id=\"mp_config_link\" href=\"{$dataUrl}\" target=\"_blank\" style=\"text-decoration: underline; font-weight: 700;\">{$dataUrl}</a>"
                        ),
                        array(
                            'type' => 'text',
                            'label' => $this->l('Client Id'),
                            'name' => 'client_id',
                            'class' => 'fixed-width-lg'
                        ),
                        array(
                            'type' => 'text',
                            'label' => $this->l('Client Secret'),
                            'name' => 'client_secret',
                            'class' => 'fixed-width-xxl',
                        ),
                    ),
                    'desc' =>  $this->l('Currently selected currency for this module: ')."<b>{$currentCurrency->name} ({$currentCurrency->iso_code})</b>"
                )
            ),
            array(
                'form' => array(
                    'legend' => array(
                        'title' => '2. '.$this->l('Payment')
                    ),
                    'input' => array(
                        array(
                            'type' => 'checkbox',
                            'label' => $this->l('Payment methods you wish to accept'),
                            'name' => 'payment',
                            'values' => array(
                                'query' => array(
                                    array('value' =>'credit_card', 'text' => $this->l('Credit Card')),
                                    array('value' =>'debit_card', 'text' => $this->l('Debit Card')),
                                    array('value' =>'ticket', 'text' => $this->l('Ticket')),
                                    array('value' =>'bank_transfer', 'text' => $this->l('Bank Transfer')),
                                    array('value' =>'atm', 'text' => $this->l('ATM Bank Transfer')),
                                ),
                                'id' => 'value',
                                'name' => 'text'
                            ),
                        ),
                        array(
                            'type' => 'text',
                            'label' => $this->l('Additional fee'),
                            'name' => 'fee',
                            'class' => 'fixed-width-md',
                            'desc' => $this->l('Additional fee you want to charge customers who choose this payment option. Can be either a fixed amount (without the percent sign) or a percentage (ending with the percent sign). Set to either 0.00 or 0.00% to disable.')
                        ),
                    )
                )
            ),
            array(
                'form' => array(
                    'legend' => array(
                        'title' => '3. '.$this->l('Checkout')
                    ),
                    'input' => array(
                        array(
                            'type' => 'radio',
                            'label' => $this->l('Client Checkout Type'),
                            'name' => 'checkout_type',
                            'class' => 't',
                            'values' => array(
                                array(
                                    'id' => 'modal',
                                    'value' => 'modal',
                                    'label' => '<abbr title="'.$this->l('Translucent overlay on top of your store\'s page').'">'.$this->l('Modal Window').'</abbr>'
                                ),
                                /*array(
                                    'id' => 'iframe',
                                    'value' => 'iframe',
                                    'label' => '<abbr title="'.$this->l('Checkout looks like it\'s inside your store').'">'.$this->l('Integrated iframe').'</abbr>'
                                ),*/
                                array(
                                    'id' => 'redirect',
                                    'value' => 'redirect',
                                    'label' => '<abbr title="'.$this->l('User is temporarily redirected to Mercadopago\'s server').'">'.$this->l('Redirection').'</abbr>'
                                ),
                            ),
                        ),
                        array(
                            'type' => _PS_VERSION_ < 1.6 ? 'radio' : 'switch',
                            'label' => $this->l('Enable sandbox mode'),
                            'name' => 'sandbox',
                            'hint' => $this->l('Only for testing, normal payments won\'t work until disabled'),
                            'class' => 't',
                            'is_bool' => TRUE,
                            'values' => array(
                                array(
                                    'id' => 'sandbox_on',
                                    'value' => '1',
                                    'label' => 'on',
                                ),
                                array(
                                    'id' => 'sandbox_off',
                                    'value' => '0',
                                    'label' => 'off',
                                    'p' => $this->l('Only for testing, normal payments won\'t work until disabled')
                                ),
                            ),
                            'desc' => $this->l('Sandbox mode allows you to test Mercadopago\'s checkout using simulated payments, without using your own credit cards or Mercadopago account balance. You can check how to use it when making test orders to generate specific payment statuses at the following URL (opens in a new window):')."<a id=\"mp_config_link\" href=\"https://developers.mercadopago.com/sandbox?lang={$doclang}#sandbox-ip-examples\" target=\"_blank\" style=\"text-decoration: underline; font-weight: 700;\">https://developers.mercadopago.com/sandbox?lang={$doclang}#sandbox-ip-examples</a>"
                        ),
                    )
                )
            ),
            array(
                'form' => array(
                    'legend' => array(
                        'title' => '4. '.$this->l('Integration')
                    ),
                    'description' => $notifBox,
                    'input' => array(
                        array(
                            'type' => 'select',
                            'label' => $this->l('Pending Order Status'),
                            'name' => 'wait_status',
                            'options' => array(
                                'query' => array_merge(array(
                                    array('value' =>'0', 'text' => $this->l('Select a status for pending orders')),
                                    array('value' =>'new', 'text' =>'=='.$this->l('Create a new status').'=='),
                                ), $selectableStatuses),
                                'id' => 'value',
                                'name' => 'text'
                            ),
                            'desc' => $this->l('This is the status that will be assigned to incoming orders marked as "pending". You can create a new status by selecting "Create a new status" and naming it something like "Awaiting Mercadopago Payment"')
                        ),
                        array(
                            'type' => 'text',
                            'label' => $this->l('Status Name'),
                            'name' => 'wait_status_name',
                            'class' => 'fixed-width-xxl',
                        ),
                        array(
                            'type' => 'color',
                            'label' => $this->l('Status Color'),
                            'name' => 'color',
                            'size' => 16,           //PS 1.5
                        ),
                        array(
                            'type' => 'free',
                            'name' => 'guijs'
                        )
                    ),
                    'submit' => array(
                        'name' => 'submitMercadopago',
                        'title' => $this->l('Save'),
                        'class' => _PS_VERSION_ < 1.6 ? 'button' : NULL
                    )
                )
            )
        );
        $fields_value = array_merge(
            array(
                'country' => $currentCountry,
                'client_id' => Tools::getValue('client_id', Configuration::get(self::CONFIG_PREFIX.'_CLIENT_ID')),
                'client_secret' => Tools::getValue('client_secret', Configuration::get(self::CONFIG_PREFIX.'_CLIENT_SECRET')),
                'fee' => Tools::getValue('fee', Configuration::get(self::CONFIG_PREFIX.'_FEE')),
                'checkout_type' => Tools::getValue('checkout_type', Configuration::get(self::CONFIG_PREFIX.'_CHECKOUT_TYPE')),
                'sandbox' => Tools::getValue('sandbox', Configuration::get(self::CONFIG_PREFIX.'_SANDBOX_MODE')),
                'wait_status' => $waitStatus,
                'wait_status_name' => '',
                'color' => '#ffff00',
                'guijs' => $guijs
            ),
            count($payments) ? array_combine(array_map(create_function('$v', 'return "payment_".$v;'), $payments), array_fill(0, count($payments), 1)) : array()
        );

        $helper = new HelperForm();

        $helper->module = $this;
        $helper->title = $this->displayName;
        $helper->name_controller = $this->name;
        $helper->identifier = $this->identifier;
        $helper->token = _PS_VERSION_ < 1.4 ? Tools::getValue('token') : Tools::getAdminTokenLite('AdminModules');
        $helper->currentIndex = _PS_VERSION_ < '1.5' ? "index.php?tab=AdminModules&configure={$this->name}" : $this->context->link->getAdminLink('AdminModules', false).'&configure='.$this->name.'&tab_module='.$this->tab.'&module_name='.$this->name;
        $lang = new Language((int)Configuration::get('PS_LANG_DEFAULT'));
        $helper->default_form_language = $lang->id;
        $helper->submit_action = '';
        $helper->fields_value = $fields_value;

        return $header.$helper->generateForm($fields_form);
    }
    
    protected function _getErrors()
    {
        $error = '';
        if (!is_numeric(trim(Tools::getValue('client_id'))) || Tools::getValue('client_id')==0)
            $error .= $this->displayError($this->l('Invalid Client Id'));
        if (!Tools::getValue('client_secret'))
            $error .= $this->displayError($this->l('Invalid Secret'));
        if (!count($this->toolGetPayments()))
            $error .= $this->displayError($this->l('You must select at least one payment type'));
        if ((!is_numeric(Tools::getValue('wait_status')) && !Tools::getValue('wait_status_name')) || (is_numeric(Tools::getValue('wait_status')) && Tools::getValue('wait_status')<1))
            $error .= $this->displayError($this->l('Invalid Pending Order Status'));
        if (!preg_match('/^[\d]*[\.]?[\d]*%?$/', Tools::getValue('fee')) OR Tools::getValue('fee') == '%')
            $error .= $this->displayError($this->l('Additional fee format is incorrect. Should be either an amount, such as 7.50, or a percentage, such as 6.99%'));
        return $error;
    }
    
    protected function _saveConfig()
    {
        $extraMessage = '';
        $countryCurrencyId = Currency::getIdByIsoCode($this->countryCurrency[Tools::getValue('country')]);
        if (!empty($countryCurrencyId) && $this->getCurrency()->id != $countryCurrencyId) {           //If currency differs from module's country setting, attempt to adjust it automatically
            $countryCurrency = New Currency($countryCurrencyId);
            ob_start(); ?>
                DELETE FROM <?php echo pSQL(_DB_PREFIX_);?>module_currency
                WHERE `id_module` = <?php echo (int)$this->id; ?>
            <?php Db::getInstance()->Execute(ob_get_clean());
            ob_start(); ?>
                INSERT INTO <?php echo pSQL(_DB_PREFIX_);?>module_currency (`id_currency`, `id_module`)
                VALUES (<?php echo (int)$countryCurrencyId; ?>, <?php echo (int)$this->id; ?>)
            <?php $extraMessage = Db::getInstance()->Execute(ob_get_clean())  ?  '<br />'.$this->l('This module\'s currency has been changed automatically to ').$countryCurrency->name  :  '';
        }
        $waitStatus = Tools::getValue('wait_status');
        if ($name = Tools::getValue('wait_status_name')) {          //Create a new Order State for pending orders
            $names = array();
            $templates = array();
            foreach (Language::getLanguages() as $lang) {
                $names[$lang['id_lang']] = $name;
                $templates[$lang['id_lang']] = 'waiting_for_payment';
            }
            $orderState = new OrderState();
            $orderState->name = $names;
            $orderState->template = $templates;
            $orderState->send_email = FALSE;
            $orderState->invoice = FALSE;
            $orderState->color = Tools::getValue('color');
            $orderState->unremovable = FALSE;
            $orderState->logable = FALSE;
            $orderState->delivery = FALSE;
            $orderState->hidden = FALSE;
            if ($orderState->add()) {
                $waitStatus = $orderState->id;
                $extraMessage .= '<br />'.$this->l('New pending order status created')." ID: $waitStatus";
                $this->wait_status = $waitStatus;
            }
        }
        Configuration::updateValue(self::CONFIG_PREFIX.'_COUNTRY', Tools::getValue('country'));
        Configuration::updateValue(self::CONFIG_PREFIX.'_CLIENT_ID', trim(Tools::getValue('client_id')));
        Configuration::updateValue(self::CONFIG_PREFIX.'_CLIENT_SECRET', trim(Tools::getValue('client_secret')));
        Configuration::updateValue(self::CONFIG_PREFIX.'_WAIT_STATUS', $waitStatus);
        Configuration::updateValue(self::CONFIG_PREFIX.'_PAYMENT_METHODS', Tools::jsonEncode($this->toolGetPayments()));
        Configuration::updateValue(self::CONFIG_PREFIX.'_CHECKOUT_TYPE', Tools::getValue('checkout_type'));
        Configuration::updateValue(self::CONFIG_PREFIX.'_SANDBOX_MODE', Tools::getValue('sandbox') ? 1 : 0);
        Configuration::updateValue(self::CONFIG_PREFIX.'_FEE', Tools::getValue('fee'));
        return $this->displayConfirmation($this->l('Configuration saved') . $extraMessage);
    }
    
    
    public function hookPayment($params)
    {
        //Return False unless there are no warnings or the only warning is about sandbox mode
        if (!$this->active OR count($this->fatalWarnings))
            return FALSE;
        if (Configuration::get(self::CONFIG_PREFIX.'_CHECKOUT_TYPE') == 'redirect') {
            $btext2 = '(' . $this->l('your order will be redirected to Mercadopago\'s payment system') . ')';
        }
        $requireFee = (bool)($this->getTotalWithFee(1, Configuration::get(self::CONFIG_PREFIX.'_FEE'))>1);
        $feeamount = strpos(Configuration::get(self::CONFIG_PREFIX.'_FEE'), '%') ? Configuration::get(self::CONFIG_PREFIX.'_FEE') : $this->context->currency->sign . Configuration::get(self::CONFIG_PREFIX.'_FEE');
        $feetext = $this->l('An extra fee of') . " $feeamount " . $this->l('will apply to payments using this method');
        $mpCountry = Configuration::get(self::CONFIG_PREFIX.'_COUNTRY');
        $this->context->smarty->assign(  array(
            'buttonText' => ($this->l('Pay with Mercadopago ') ." ". $this->countryName[Configuration::get(self::CONFIG_PREFIX.'_COUNTRY')]),
            'buttonText2' => isset($btext2) ? $btext2 : '',
            'mpCountry' => $mpCountry,
            'paymentPath' => (method_exists('Tools','getShopDomainSsl') ? Tools::getShopDomainSsl(true, true) : Tools::getHttpHost(true, true))
                .(_PS_VERSION_ < '1.5' ? $this->_path.'payment.php' : __PS_BASE_URI__."index.php?fc=module&module={$this->name}&controller=payment"),
            'imagePath' => (method_exists('Tools','getShopDomainSsl') ? Tools::getShopDomainSsl(true, true) : Tools::getHttpHost(true, true)).$this->_path.'views/img/',
            'imageName' => (isset(MpOverride::$imageFile) && Tools::strlen(MpOverride::$imageFile) && file_exists(_PS_MODULE_DIR_."{$this->name}/views/img/".MpOverride::$imageFile)) ? MpOverride::$imageFile : "mercadopago_{$mpCountry}.png",
            'sandboxMode' => Configuration::get(self::CONFIG_PREFIX.'_SANDBOX_MODE') ? $this->l('Sandbox mode on, only for testing') : FALSE,
            'mpFee' => $requireFee ? $feetext : '',
            'this_name' => $this->name
        ) );
        return $this->display(__FILE__, _PS_VERSION_ < 1.5 ? 'views/templates/hook/payment.tpl' : 'payment.tpl');
    }
    
    public function hookPaymentReturn($params)
    {
        if (!$this->active)
			return FALSE;
		return $this->display(__FILE__, _PS_VERSION_ < 1.5 ? 'views/templates/hook/payment_return.tpl' : 'payment_return.tpl');
    }
    
    public function hookInvoice($params)
    {
        $order = New Order($params['id_order']);
        if ($order->module!=$this->name)
            return FALSE;
        
        require_once 'API/mercadopagoAPI.php';
        $transactionId = Db::getInstance()->getValue('SELECT `transaction_id` FROM `' . pSQL(_DB_PREFIX_.self::DB_TABLE) . '` WHERE `order_id` = ' . (int)$order->id);
        if (!Tools::strlen($transactionId)) {
            $transactionError = $this->l('Error: No Mercadopago transaction data available for this order');
        } else {
            try {
                $collection = MercadopagoInterconnector::queryTransaction($transactionId, $this->_getUpdatedToken(), TRUE, Configuration::get(self::CONFIG_PREFIX . '_SANDBOX_MODE'));
            } catch (Exception $e) {
                return false;
            }
            switch ($collection['status']) {
                case 'pending':
                    $status = $this->l('Pending');
                    break;
                case 'approved':
                    $status = $this->l('Approved');
                    break;
                case 'in_process':
                    $status = $this->l('In process');
                    break;
                case 'rejected':
                    $status = $this->l('Rejected');
                    break;
                case 'cancelled':
                    $status = $this->l('Cancelled');
                    break;
                case 'refunded':
                    $status = $this->l('Refunded');
                    break;
                case 'in_mediation':
                    $status = $this->l('In mediation');
                    break;
                default:
                    $status = $this->l('Error querying Mercadopago server');
                    break;
            }
            $mpId = $collection ? implode(', ', $collection['id']) : '';
            $currency = new Currency($order->id_currency);
            $this->context->currency = $currency;           //Fix PS bug with multicurrencies
            $cart = new Cart($order->id_cart);
            $fee = $this->getFieldFromTransactionId($transactionId, 'fee');
            $orderToPay = Tools::ps_round($this->getTotalWithFee($order->total_paid, $fee), 2);
            $cart_details = $cart->getSummaryDetails(null, true);
            $cartTotal = $cart_details['total_price'];
            $cartToPay = Tools::ps_round($this->getTotalWithFee($cartTotal, $fee), 2);
            $paidFee = ($collection['transaction_amount'] && abs($collection['transaction_amount'] - $orderToPay) < $this->delta) ? Tools::ps_round($collection['transaction_amount'] - $order->total_paid, 2) : max(0, Tools::ps_round($collection['transaction_amount'] - $cartTotal, 2));
        }
        $this->context->smarty->assign(  array(
            'moduleName' => self::MODULE_NAME,
            'mpTitle' => $this->l('Mercadopago Transaction Information'),
            'mpIdTitle' => $this->l('Mercadopago ID'),
            'mpTransactionTitle' => $this->l('Transaction ID'),
            'mpStatusTitle' => $this->l('Status'),
            'mpFeeTitle' => $this->l('Additional fee'),
            'mpId' => $mpId,
            'transactionId' => $transactionId,
            'currencySign' => $currency->sign,
            'transactionError' => isset($transactionError) ? $transactionError : false,
            'mpStatus' => $status,
            'mpFee' => $fee,
            'collection' => $collection,
            'amountDiscrepancy' => ($collection['transaction_amount'] && abs($collection['transaction_amount'] - $cartToPay) > $this->delta && abs($collection['transaction_amount'] - $orderToPay) > $this->delta),
            'amountDiscrepancyWarning' => $this->l('Warning: Order amount discrepancy, please check your Mercadopago account for order'),
            'totalPayedTitle' => $this->l('Total amount payed by customer:'),
            'cartOrderDiscrepancy' => ($collection['transaction_amount'] && abs($collection['transaction_amount'] - $orderToPay) > $this->delta && abs($orderToPay - $cartToPay) > $this->delta),
            'cartOrderDiscrepancyWarning' => $this->l('Warning: Discrepancy between cart and order amounts.'),
            'cartPayedTitle' => $this->l('Total amount of this orderʻs cart'),
            'carrierDiscrepancy' => ($cart->id_carrier != $order->id_carrier),
            'carrierDiscrepancyWarning' => $this->l('Warning: Cart carrier is different from order carrier.'),
            'orderToPay' => $orderToPay,
            'cartTotal' => $cartTotal,
            'cartToPay' => $cartToPay,
            'paidFee' => $paidFee,
            'cartCarrier' => $cart->id_carrier,
            'cartCarrierTitle' => $this->l('Cart carrier'),
            'orderCarrier' => $order->id_carrier,
            'orderCarrierTitle' => $this->l('Order carrier'),
            'showCollection' => (isset(MpOverride::$show_mercadopago_collection) && MpOverride::$show_mercadopago_collection),
            'collectionPrint' => print_r($collection, true),
        ) );
        return $this->display(__FILE__, _PS_VERSION_ < '1.6' ? (_PS_VERSION_ < '1.5' ? 'views/templates/hook/invoice15.tpl' :'invoice15.tpl') : 'invoice.tpl');
    }
    
    
    public function process()
    {
        $currency = $this->getCurrency();
        if ((int)$this->context->currency->id != (int)$currency->id) {
            $this->context->cookie->id_currency = $this->context->cart->id_currency = $this->context->currency->id = $currency->id;
            $this->context->cart->update();
            Tools::redirect(_PS_VERSION_ < '1.5' ? "modules/{$this->name}/payment.php" : "index.php?fc=module&module={$this->name}&controller=payment");
        }
        $this->transactionId = Tools::substr( abs(ip2long($_SERVER['REMOTE_ADDR'])), 0, 10 ) . time();    //create a unique ID based on the current time and customer's IP address
        $this->storeTransaction($this->transactionId);
        require_once 'API/mercadopagoAPI.php';
        if (!$this->context->cart->id)
            Tools::redirect(_PS_VERSION_ < '1.5' ? 'order.php' : 'index.php?controller=order');
        $baseUrl = (method_exists('Tools','getShopDomainSsl') ? Tools::getShopDomainSsl(true, true) : Tools::getHttpHost(true, true)) . __PS_BASE_URI__;    //1.3.x Hacks
        $shopCurrency = $this->getCurrency($this->context->cart->id_currency);
        $currentCustomer = New Customer((int)$this->context->cart->id_customer);
        $itemName = $this->l('Order from').' '.Configuration::get('PS_SHOP_NAME');
        if (Tools::strlen($itemName) > 100)
            $itemName = Tools::substr($itemName, 0, 100);
        if (method_exists('Cache','clean')) {   //Reduce chance of last-minute shipping and wrapping change discrepancies
            Cache::clean('objectmodel_Cart_' . (int)$this->context->cart->id . '_0_0');
        }
        $cart = new Cart($this->context->cart->id);     //Refresh cart
        if (method_exists('Cart','getDeliveryOption')) {     //Carrier discrepancy bug workaround
            $deliveryOption = $cart->getDeliveryOption();
            if (!is_null($cart->id_address_delivery) && isset($deliveryOption[$cart->id_address_delivery]) && Tools::strlen($deliveryOption[$cart->id_address_delivery]) && !in_array($cart->id_carrier, ($exp = explode(',', $deliveryOption[$cart->id_address_delivery])))) {
                $cart->id_carrier = $exp[0];
                $cart->update();
            }
        }
        $cart_details = $cart->getSummaryDetails(null, true);
        $price = $cart_details['total_price'];
        if (!$shopCurrency->decimals)
            $price = Tools::ps_round((float)$price);
        $products = $cart->getProducts();
        $productsNum = count($products);
        $requireFee = (bool)($this->getTotalWithFee(1, Configuration::get(self::CONFIG_PREFIX.'_FEE'))>1);
        $items = array();
        for ($i=1; $i<=$productsNum; $i++) {    //Add all items to mercadopago cart
            array_push($items,
                ( !empty($products[$i-1]['legend']) ? $products[$i-1]['legend'] : $products[$i-1]['name']  .  (!empty($products[$i-1]['attributes_small']) ? (' ' . $products[$i-1]['attributes_small']) : '') . '( x' . $products[$i-1]['cart_quantity'] . ')' )
            );
        }
        $price = $requireFee ? Tools::ps_round($this->getTotalWithFee($price, Configuration::get(self::CONFIG_PREFIX.'_FEE')), 2) : $price;
        try {
            $mpUrl = MercadopagoInterconnector::getPaymentLink(
                $this->_getUpdatedToken(),
                array(
                    'items' => array(array(
                        'title' => $itemName,
                        'quantity' => 1,
                        'unit_price' => (float)$price,
                        'currency_id' => Tools::strtoupper($this->context->currency->iso_code),
                        'picture_url' => file_exists(_PS_IMG_DIR_.Configuration::get('PS_LOGO')) ? $baseUrl.'img/'.Configuration::get('PS_LOGO') : '',
                        'id' => $this->context->cart->id,
                        'description' => implode('; ', $items)
                    )),
                    'payer' => array(
                        'name' => $currentCustomer->firstname,
                        'surname' => $currentCustomer->lastname,
                        'email' => $currentCustomer->email
                    ),
                    'back_urls' => array(
                        'success' => $baseUrl . (_PS_VERSION_ < '1.5' ? "modules/{$this->name}/validation.php?cid={$this->context->cart->id}&tid={$this->transactionId}" : "index.php?fc=module&module={$this->name}&controller=validation&cid={$this->context->cart->id}&tid={$this->transactionId}"),
                        'failure' => $baseUrl . (_PS_VERSION_ < '1.5' ? (_PS_VERSION_ < '1.4' ? 'order.php' : 'order-opc.php') : 'index.php?controller=order-opc'),
                        'pending' => $baseUrl . (_PS_VERSION_ < '1.5' ? "modules/{$this->name}/validation.php?cid={$this->context->cart->id}&tid={$this->transactionId}" : "index.php?fc=module&module={$this->name}&controller=validation&cid={$this->context->cart->id}&tid={$this->transactionId}")
                    ),
                    'payment_methods' => array(
                        'excluded_payment_types' => $this->_getExcludedPayments(),
                        'excluded_payment_methods' => (isset(MpOverride::$excludedPaymentMethods) && count(MpOverride::$excludedPaymentMethods)) ? $this->formatExcludedPayments(MpOverride::$excludedPaymentMethods) : array()
                    ),
                    'external_reference' => $this->transactionId,
                ),
                Configuration::get(self::CONFIG_PREFIX . '_SANDBOX_MODE')
            );
        } catch(Exception $e) {
            Logger::addLog('Payment link request error, cart: '.$this->context->cart->id, 3);
            die(_PS_MODE_DEV_ ? $e->getMessage() : $this->l('Payment link request error'));
        }
        return array(
            'mpType' => Configuration::get(self::CONFIG_PREFIX.'_CHECKOUT_TYPE'),
            'loadingText' => $this->l('Please wait, Mercadopago\'s payment overlay is being loaded').'...',
            'loaderText' => $this->l('Loading').'...',
            'mpOverlay' => $this->l('Mercadopago checkout overlay in use'),
            'mpRedirecting' => $this->l('Please wait, redirecting').'...',
            'mpInit' => $mpUrl,
            'mpBase' =>  $baseUrl,
            'mpForward' => $baseUrl . (_PS_VERSION_ < '1.5' ? "modules/{$this->name}/validation.php?cid={$this->context->cart->id}&tid={$this->transactionId}" : "index.php?fc=module&module={$this->name}&controller=validation&cid={$this->context->cart->id}&tid={$this->transactionId}"),
            'mpBack' => $baseUrl . (_PS_VERSION_ < '1.5' ? (_PS_VERSION_ < '1.4' ? 'order.php' : 'order-opc.php') : 'index.php?controller=order-opc')
        );

    }

    /**
     * Displays checkout template for Prestashop < 1.5
     * @param $smartyData
     * @return string
     */
    public function displayCheckoutOld($smartyData) {
        if (_PS_VERSION_ < '1.5') {
            $this->context->smarty->assign($smartyData);
            return $this->display(__FILE__, 'views/templates/front/confirmation.tpl');
        }
    }

    /**
     * @param $transactionId
     * @return bool
     */
    public function storeTransaction($transactionId)    //store transaction details in DB
    {
        ob_start(); ?>
            REPLACE INTO `<?php echo pSQL(_DB_PREFIX_.self::DB_TABLE);?>` (`transaction_id`, `cart_id`)
            VALUES ("<?php echo pSQL($transactionId); ?>", <?php echo (int)$this->context->cart->id; ?>)
        <?php return Db::getInstance()->Execute(ob_get_clean());
    }

    /**
     * @param $cart
     * @return bool
     */
    public function assignOrderToTransaction($cart)          //store new order id in DB
    {
        if ($orderId = Order::getOrderByCartId((int)$cart->id)) {
            ob_start(); ?>
                UPDATE `<?php echo pSQL(_DB_PREFIX_.self::DB_TABLE);?>`
                SET `order_id` = <?php echo (int)$orderId; ?>,
                `fee` = "<?php echo pSQL(Configuration::get(self::CONFIG_PREFIX.'_FEE')); ?>"
                WHERE `cart_id` = <?php echo (int)$cart->id; ?>
            <?php return Db::getInstance()->Execute(ob_get_clean());
        }
        return FALSE;
    }

    /**
     * @param string $transactionId
     * @param string $field     Options: order_id, cart_id
     * @return string
     */
    public function getFieldFromTransactionId($transactionId, $field)
    {
        ob_start(); ?>
            SELECT `<?php echo pSQL($field);?>`
            FROM `<?php echo pSQL(_DB_PREFIX_.self::DB_TABLE);?>`
            WHERE `transaction_id` = "<?php echo pSQL($transactionId); ?>"
        <?php return Db::getInstance()->getValue(ob_get_clean());
    }


    /**
     * @param $orderId
     * @return array
     */
    public function queryTransactionData($orderId)
    {
        require_once 'API/mercadopagoAPI.php';
        $json = MercadopagoInterconnector::queryTransaction($orderId, $this->_getUpdatedToken(), FALSE, Configuration::get(self::CONFIG_PREFIX.'_SANDBOX_MODE')) OR die('Error retrieving transaction data');
        return $json;
    }

    /**
     * Apply a fee to a payment amount
     * @param float $netAmount
     * @param string $fee (float with optional percent sign at the end)
     */
    public function getTotalWithFee($netAmount, $fee)
    {
        $fee = Tools::strlen($fee) ? $fee : '0';
        return strpos($fee, '%') ? $netAmount*(1+(float)Tools::substr($fee, 0, -1)/100) : $netAmount+(float)$fee;
    }

    /**
     * Works like Tools::getValue for HelperForm payment checkbox group
     * @return array
     */
    public function toolGetPayments()
    {
        $payments = array();
        foreach ($this->paymentsArray as $paym) {
            if (Tools::getValue('payment_'.$paym, 0)) {
                $payments[] = $paym;
            }
        }
        if (!Tools::isSubmit('submitMercadopago') && !count($payments)) {
            $payments = $this->_getPaymentConfig();
        }
        return $payments;
    }


    /**
     * Returns up to date access token
     * @param bool $force
     * @throws Exception
     * @return string
     */
    protected function _getUpdatedToken($force=FALSE)
    {
        $time = getdate();
        $time = $time[0];
        if (!Tools::strlen(Configuration::get(self::CONFIG_PREFIX.'_TOKEN_KEY'))
        OR $time >= Configuration::get(self::CONFIG_PREFIX.'_TOKEN_DATE')
        OR $force) {
            $data = MercadopagoInterconnector::retrieveAccessToken(array(
                'id' => Configuration::get(self::CONFIG_PREFIX.'_CLIENT_ID'),
                'secret' => Configuration::get(self::CONFIG_PREFIX.'_CLIENT_SECRET')
            ));
            if (!$data) {
                throw new Exception($this->l('Unable to connect to Mercadopago authentication server, please check your server can initiate outbound connections.'));
            } elseif (isset($data['error']) || !isset($data['access_token'])) {
                throw new Exception($this->l('There seems to be an error in your Client Id or your Client Secret'));
            }
            Configuration::updateValue(self::CONFIG_PREFIX.'_TOKEN_KEY', $data['access_token']);
            Configuration::updateValue(self::CONFIG_PREFIX.'_TOKEN_DATE', $time+$data['expires_in']);
            return $data['access_token'];
        } else
            return Configuration::get(self::CONFIG_PREFIX.'_TOKEN_KEY');
    }

    /*
     *  $config = '["account_money","credit_card","debit_card","ticket","bank_transfer","atm"]'
     */
    protected function _getPaymentConfig()
    {
        return Tools::jsonDecode(Configuration::get(self::CONFIG_PREFIX.'_PAYMENT_METHODS'));
    }

    protected function _getExcludedPayments()
    {
        $payments = array_diff($this->paymentsArray, $this->_getPaymentConfig());
        return $this->formatExcludedPayments($payments);
    }

    protected function formatExcludedPayments($payments)
    {
        $formattedPayments = array();
        foreach ($payments as $paym) {
            $formattedPayments[] = array('id' => $paym);
        }
        return $formattedPayments;
    }
}

?>